/**
* \file: WaylandContext.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Digital iPod Out - Wayland Adapter
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <cstring>
#include <poll.h>
#include <sys/prctl.h>
#include <sys/eventfd.h>
#include <adit_logging.h>
#include "WaylandContext.h"

LOG_IMPORT_CONTEXT(dipo_wl);

namespace adit { namespace carplay
{
using namespace std;

WaylandContext::DeviceListener::~DeviceListener()
{
}

WaylandContext::WaylandContext():wlSeatList(),display(nullptr),shareWlDisplay(false),inputQueue(nullptr),compositor(nullptr),
        shm(nullptr),shmFormat(0),registry(nullptr),adapterContext(nullptr),touchListener(nullptr),
        pointerListener(nullptr),threadRunning(false),threadId(0),processIntervalUSec(0),
        shutDownEventFd(-1){staticInitialize();pthread_mutex_init(&SeatLock, NULL);}

WaylandContext::WaylandContext(struct wl_display* inDisplay):wlSeatList(),display(inDisplay),shareWlDisplay(false),inputQueue(nullptr),
        compositor(nullptr),shm(nullptr),shmFormat(0),registry(nullptr),adapterContext(nullptr),
        touchListener(nullptr),pointerListener(nullptr),threadRunning(false),threadId(0),
        processIntervalUSec(0),shutDownEventFd(-1){staticInitialize();}

WaylandContext::~WaylandContext()
{
    StopInputThread();

    LOGD_DEBUG((dipo_wl, "destroy Wayland context"));

    std::list<WLSeat*>::iterator it;
    pthread_mutex_lock(&SeatLock);
    for(it = wlSeatList.begin(); it != wlSeatList.end();it++)
    {
        delete(*it);
    }
    wlSeatList.clear();
    pthread_mutex_unlock(&SeatLock);

    dipo_safe_call(wl_event_queue_destroy, inputQueue);
    dipo_safe_call(wl_compositor_destroy, compositor);
    dipo_safe_call(wl_registry_destroy, registry);
    dipo_safe_call(wl_shm_destroy, shm);

    if (!shareWlDisplay)
    {
        if (adapterContext)
        {
            compositor_shim_terminate(adapterContext);
        }

        dipo_safe_call(wl_display_disconnect, display);
    }

    pthread_mutex_destroy(&SeatLock);
}

bool WaylandContext::Initialize()
{

    int wlError = -1;
    /*
     * Connect to a Wayland display
     * If name is NULL, its value will bee replaced with the
     * WAYLAND_DISPLAY environment variable if it is set,
     * otherwise display "wayland-0" will be used.
     */
    if (!shareWlDisplay)
    {
        display = wl_display_connect(nullptr);
        if (display == nullptr)
        {
            LOG_ERROR((dipo_wl, "wl_display_connect failed"));
            return false;
        }
    }
    LOGD_DEBUG((dipo_wl, "after wl_display_connect"));

    registry = wl_display_get_registry(display);
    if (registry == nullptr)
    {
        LOG_ERROR((dipo_wl, "wl_display_get_registry failed"));
        return false;
    }

    wlError = wl_registry_add_listener(registry, &registryListener, this);
    if (wlError < 0)
    {
        LOG_ERROR((dipo_wl, "wl_registry_add_listener failed %d", wlError));
        return false;
    }

    inputQueue = wl_display_create_queue(display);
    if (inputQueue == nullptr)
    {
        LOG_ERROR((dipo_wl, "failed to create a queue for input events"));
        return false;
    } else {
        // Registry events will come to inputQueue
        wl_proxy_set_queue((struct wl_proxy*) registry, inputQueue);
    }

    // wait for Registry Global event
    wlError = wl_display_roundtrip_queue(display, inputQueue);
    if (wlError < 0)
    {
        LOG_ERROR((dipo_wl, "First wl_display_roundtrip failed %d", wlError));
        return false;
    }

    /* block until callback returns and connectionId is set */
    wlError = wl_display_roundtrip_queue(display, inputQueue);
    if (wlError < 0)
    {
        LOG_ERROR((dipo_wl, "Second wl_display_roundtrip failed %d", wlError));
        return false;
    }

    if (!shareWlDisplay)
    {
        adapterContext = compositor_shim_initialize(display);
        if (!adapterContext)
        {
            LOG_ERROR((dipo_wl, "compositor_shim_initialize failed"));
            return false;
        }
    }

    return true;
}

void WaylandContext::SetTouchListener(struct wl_touch_listener* inListener,
        unique_ptr<DeviceListener> inObj)
{
    touchListener = inListener;
    touchListenerObj = move(inObj);
}

void WaylandContext::SetPointerListener(struct wl_pointer_listener* inListener,
        unique_ptr<DeviceListener> inObj)
{
    pointerListener = inListener;
    pointerListenerObj = move(inObj);
}

bool WaylandContext::StartInputThread(uint32_t inProcessIntervalUSec)
{
    if (threadRunning)
    {
        LOG_ERROR((dipo_wl, "WaylandContext input thread is already running"));
        return false;
    }

    /* create eventFd to send stop event */
    shutDownEventFd = eventfd(0, 0);
    if (shutDownEventFd < 0) {
        LOG_ERROR((dipo_wl, "Failed to create shutdown eventFd, error=%d", errno));
        return false;
    }

    threadRunning = true;
    processIntervalUSec = inProcessIntervalUSec;

    int ret = pthread_create(&threadId, NULL, inputThread, this);
    if (ret != 0)
    {
        threadRunning = false;
        LOG_ERROR((dipo_wl, "pthread_create failed: %d %s", errno, strerror(errno)));
        return false;
    }

    return true;
}

void WaylandContext::StopInputThread()
{
    if (threadRunning)
    {
        threadRunning = false;

        if (shutDownEventFd >= 0) {
            // trigger shutdown event
            if (eventfd_write(shutDownEventFd, shutDownEvent) != 0){
                LOG_ERROR((dipo_wl, "Failed to trigger input thread shutdown event"));
            }
        } else {
            LOG_ERROR((dipo_wl, "shutDownEventFd is invalid"));
        }

        int ret = pthread_join(threadId, nullptr);
        if (ret != 0)
        {
            // if pthread_join fails just log the error using errno
            LOG_ERROR((dipo_wl, "pthread_join failed with error %d %s", errno, strerror(errno)));
        }

        if (shutDownEventFd >= 0) {
            close(shutDownEventFd);
            shutDownEventFd = -1;
        }
    }
}

// ========== private methods ==========

struct wl_registry_listener WaylandContext::registryListener;
struct wl_shm_listener WaylandContext::shmListener;
struct wl_seat_listener WaylandContext::seatListener;
bool WaylandContext::staticInitialized = false;

void* WaylandContext::inputThread(void* inData)
{
    WaylandContext* me = static_cast<WaylandContext*>(inData);
    dipo_return_value_on_invalid_argument(dipo_wl, me == nullptr, nullptr);

    // set thread name
    prctl(PR_SET_NAME, "WaylandTouch", 0, 0, 0);

    LOGD_DEBUG((dipo_wl, "input poll thread started"));

    //wl_display_flush(me->display);
    while (me->threadRunning)
    {
        int wlError = -1;

        while (wl_display_prepare_read_queue(me->display, me->inputQueue) != 0)
        {
            // dispatch events which might be read already
            wlError = wl_display_dispatch_queue_pending(me->display, me->inputQueue);
            if (wlError < 0)
            {
                LOG_ERROR((dipo_wl, "Dispatching events, which are already read, failed "
                        "errno: %d - Exit input thread", errno));
                return nullptr;
            }
        }

        wlError = wl_display_flush(me->display);
        if (wlError < 0)
        {
            LOG_ERROR((dipo_wl, "Wayland flush failed errno: %d - Exit input thread",
                    errno));
            wl_display_cancel_read(me->display);
            return nullptr;
        }

        // unfortunately we have to poll for input events
        struct pollfd pfd[2];
        // shutdown events
        pfd[event_shutdown].fd = me->shutDownEventFd;
        pfd[event_shutdown].events = me->shutDownEvent;
        // input events
        pfd[event_input].fd = wl_display_get_fd(me->display);
        pfd[event_input].events = POLLIN;

        int retval = poll(pfd, 2, -1);
        if(retval <= 0)
        {
            wl_display_cancel_read(me->display);

            // handle shut down event before Wayland event
        } else if (pfd[event_shutdown].revents & me->shutDownEvent) {

            wl_display_cancel_read(me->display);
            break;

        } else if (pfd[event_input].revents & POLLIN) {

            wlError = wl_display_read_events(me->display);
            if (wlError < 0)
            {
                LOG_ERROR((dipo_wl, "Reading wayland event failed errno: %d", errno));
            }

            wlError = wl_display_dispatch_queue_pending(me->display, me->inputQueue);
            if (wlError < 0)
            {
                LOG_ERROR((dipo_wl, "Dispatching the input queue failed errno: %d", errno));
            }
        }
    }

    LOGD_DEBUG((dipo_wl, "input poll thread ended"));
    return nullptr;
}

void WaylandContext::staticInitialize()
{
    if (!staticInitialized)
    {
        memset(&seatListener, 0, sizeof(struct wl_seat_listener));
        seatListener.capabilities = onSeatCapabilities;
        seatListener.name = onSeatName;

        memset(&shmListener, 0, sizeof(wl_shm_listener));
        shmListener.format = onShmFormats;

        memset(&registryListener, 0, sizeof(wl_registry_listener));
        registryListener.global = onRegistry;
        registryListener.global_remove = onRegistryRemove;

        staticInitialized = true;
    }
}

void WaylandContext::onShmFormats(void* inMe, struct wl_shm* inShm,
        uint32_t inFormat)
{
    (void)inShm;
    dipo_return_on_invalid_argument(dipo, inMe == nullptr);

    auto me = static_cast<WaylandContext*>(inMe);

    if (inFormat == WL_SHM_FORMAT_C8)
    {
        me->shmFormat = WL_SHM_FORMAT_C8;
    }
}

void WaylandContext::onRegistry(void* inMe, struct wl_registry* inRegistry, uint32_t inName,
        const char* inInterface, uint32_t inVersion)
{
    (void)inVersion;
    dipo_return_on_invalid_argument(dipo_wl, inMe == nullptr);
    dipo_return_on_invalid_argument(dipo_wl, inRegistry == nullptr);

    auto me = static_cast<WaylandContext*>(inMe);

    if (!me->shareWlDisplay)
    {
        if (!strcmp(inInterface, "wl_compositor"))
        {
            me->compositor = static_cast<struct wl_compositor*>(
                    wl_registry_bind(inRegistry, inName, &wl_compositor_interface, 1));
            if (me->compositor == nullptr)
            {
                LOG_ERROR((dipo_wl, "wl_registry_bind failed for wl_registry_bind"));
            }
        }

        if (!strcmp(inInterface, "wl_shm"))
        {
            me->shm = static_cast<struct wl_shm*>(
                    wl_registry_bind(inRegistry, inName, &wl_shm_interface, 1));
            if (me->shm != nullptr)
            {
                int error = wl_shm_add_listener(me->shm, &me->shmListener, me);
                if (error < 0)
                {
                    LOG_ERROR((dipo_wl, "wl_shm_add_listener failed: %d", error));
                }
            }
            else
            {
                LOG_ERROR((dipo_wl, "wl_registry_bind failed for wl_shm"));
            }
        }
    }

    if (!strcmp(inInterface, "wl_seat"))
    {
        WLSeat* pwlseat= new WLSeat();
        if(pwlseat != nullptr)
        {
            struct wl_seat* wlSeat= static_cast<struct wl_seat*>(
                    wl_registry_bind(inRegistry, inName, &wl_seat_interface, 3));
            if(wlSeat != nullptr)
            {
                int error = wl_seat_add_listener(wlSeat, &seatListener, pwlseat);
                if(error < 0 )
                {
                    LOG_ERROR((dipo_wl, "wl_seat_add_listener failed "));
                }
                /*Seat events will come to inputQueue*/
                wl_proxy_set_queue((struct wl_proxy*)wlSeat, me->inputQueue);

                pwlseat->SetWLSeat(wlSeat);
                pwlseat->SetParentWlContext(me);
                pwlseat->SetSeatID(inName);
                /*Adding WlSeat object to the list*/
                me->AddWlSeat(pwlseat);
            }
            else
            {
                LOG_ERROR((dipo_wl, "wl_registry_bind failed for wl_seat "));
            }
        }
        else
        {
            LOG_ERROR((dipo_wl, "wlSeat object allocation failed "));
        }
    }
}

void WaylandContext::onSeatCapabilities(void* inMe, struct wl_seat* inSeat, uint32_t inCaps)
{
    dipo_return_on_invalid_argument(dipo_wl, inMe == nullptr);
    dipo_return_on_invalid_argument(dipo_wl, inSeat == nullptr);

    WLSeat* pwlseat= static_cast<WLSeat*>(inMe);
    auto me = pwlseat->GetParentWlContext();
    struct wl_seat* wlSeat=pwlseat->GetWLSeat();
    if(!wlSeat)
    {
        LOG_ERROR((dipo_wl, "wlSeat not created"));
        return;
    }

    /*Touch*/
    if(me->touchListener!= nullptr && (inCaps & WL_SEAT_CAPABILITY_TOUCH))
    {
        LOGD_DEBUG((dipo_wl, "register touch listener"));
        struct wl_touch* wlTouch = pwlseat->GetWLTouch();
        if(!wlTouch )
        {
            wlTouch = wl_seat_get_touch(wlSeat);
            if(wlTouch != nullptr )
            {
                wl_touch_set_user_data(wlTouch,(void *)(me->touchListenerObj.get()));
                int wlError = wl_touch_add_listener(wlTouch,me->touchListener,(void *)(me->touchListenerObj.get()));
                if(wlError < 0)
                {
                    dipo_safe_call(wl_touch_destroy, wlTouch);
                    LOG_ERROR((dipo_wl, "wl_touch_add_listener failed"));
                }
            }
            else
            {
                LOG_ERROR((dipo_wl, "wl_seat_get_touch failed"));
            }
        }
        else
        {
            LOG_WARN((dipo_wl,"wl_touch already exists "));
        }
        pwlseat->SetWLTouch(wlTouch);
    }
    else
        {
            if(me->touchListener!=nullptr)
                LOGD_DEBUG((dipo_wl,"No touch listener to register"));
            else
                LOGD_DEBUG((dipo_wl,"No support for touch listener"));
        }


    /*Pointer*/
    if(me->pointerListener!= nullptr && (inCaps & WL_SEAT_CAPABILITY_POINTER))
    {
        LOGD_DEBUG((dipo_wl, "register touch listener"));
        struct wl_pointer* wlPointer = pwlseat->GetWLPointer();
        if(!wlPointer )
        {
            wlPointer = wl_seat_get_pointer(wlSeat);
            if(wlPointer != nullptr )
            {
                wl_pointer_set_user_data(wlPointer,(void *)(me->pointerListenerObj.get()));

                int wlError = wl_pointer_add_listener(wlPointer,me->pointerListener,(void *)(me->pointerListenerObj.get()));
                if(wlError < 0 )
                {
                    dipo_safe_call(wl_pointer_destroy, wlPointer);
                    LOG_ERROR((dipo_wl, "wl_pointer_add_listener failed"));
                }
            }
            else
            {
                LOG_ERROR((dipo_wl, "wl_seat_get_pointer failed"));
            }
        }
        else
        {
            LOG_WARN((dipo_wl,"wl_pointer already exists "));
        }
        pwlseat->SetWLPointer(wlPointer);
    }
    else
    {
        if(me->pointerListener!=nullptr)
            LOGD_DEBUG((dipo_wl,"No pointer listener to register"));
        else
            LOGD_DEBUG((dipo_wl,"No support for pointer listener"));
    }
}

void WaylandContext::onSeatName(void *inMe, struct wl_seat* inSeat, const char *inName)
{
    dipo_return_on_invalid_argument(dipo_wl, inMe == nullptr);
    dipo_return_on_invalid_argument(dipo_wl, inSeat == nullptr);

    auto pWlSeat =  static_cast<WLSeat*>(inMe);
    pWlSeat->SetSeatName(inName);

}

void WaylandContext::onRegistryRemove(void *inMe, struct wl_registry* inRegistry, uint32_t inName)
{
    (void)inRegistry;
    dipo_return_on_invalid_argument(dipo_wl, inMe == nullptr);

    auto me = static_cast<WaylandContext*>(inMe);
    std::list <WLSeat*>::iterator it;

    pthread_mutex_lock(&(me->SeatLock));
    std::list<WLSeat*> seatList = me->GetWLSeatList();
    for(it = seatList.begin(); it != seatList.end(); it++)
    {
        if ((*it)->GetSeatID() == inName )
        {
            delete(*it);
            seatList.erase(it);
        }
    }
    pthread_mutex_unlock(&(me->SeatLock));
}
} } // namespace adit { namespace carplay
